'use client'

import { useEffect, useState } from 'react'
import { useParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Formulario, CampoFormulario } from '@/lib/types'

export default function FormularioPublicoPage() {
  const params = useParams()
  const [formulario, setFormulario] = useState<Formulario | null>(null)
  const [loading, setLoading] = useState(true)
  const [submitting, setSubmitting] = useState(false)
  const [respuestas, setRespuestas] = useState<Record<string, any>>({})
  const [fechaVisita, setFechaVisita] = useState('')

  useEffect(() => {
    if (params.link) {
      fetchFormulario(params.link as string)
    }
  }, [params.link])

  async function fetchFormulario(link: string) {
    try {
      const { data, error } = await supabase
        .from('formularios')
        .select('*')
        .eq('link_publico', link)
        .eq('activo', true)
        .single()

      if (error) throw error
      setFormulario(data)
      
      // Inicializar respuestas vacías
      const initialRespuestas: Record<string, any> = {}
      data.campos?.forEach((campo: CampoFormulario) => {
        if (campo.tipo === 'checkbox') {
          initialRespuestas[campo.nombre] = false
        } else {
          initialRespuestas[campo.nombre] = ''
        }
      })
      setRespuestas(initialRespuestas)
    } catch (error) {
      console.error('Error fetching formulario:', error)
    } finally {
      setLoading(false)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    if (!formulario) return

    // Validar campos requeridos
    for (const campo of formulario.campos || []) {
      if (campo.requerido && !respuestas[campo.nombre]) {
        alert(`El campo "${campo.etiqueta}" es requerido`)
        return
      }
    }

    setSubmitting(true)

    try {
      const { error } = await supabase
        .from('respuestas_formularios')
        .insert([{
          formulario_id: formulario.id,
          datos: respuestas,
          fecha_visita: fechaVisita || null,
        }])

      if (error) throw error

      alert('¡Formulario enviado exitosamente! Gracias por tu respuesta.')
      
      // Limpiar formulario
      const initialRespuestas: Record<string, any> = {}
      formulario.campos?.forEach((campo: CampoFormulario) => {
        if (campo.tipo === 'checkbox') {
          initialRespuestas[campo.nombre] = false
        } else {
          initialRespuestas[campo.nombre] = ''
        }
      })
      setRespuestas(initialRespuestas)
      setFechaVisita('')
    } catch (error: any) {
      console.error('Error submitting formulario:', error)
      alert(`Error al enviar el formulario: ${error.message}`)
    } finally {
      setSubmitting(false)
    }
  }

  function renderCampo(campo: CampoFormulario) {
    const value = respuestas[campo.nombre] || ''

    switch (campo.tipo) {
      case 'textarea':
        return (
          <textarea
            required={campo.requerido}
            value={value}
            onChange={(e) => setRespuestas({ ...respuestas, [campo.nombre]: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={4}
            placeholder={campo.placeholder}
          />
        )
      case 'select':
        return (
          <select
            required={campo.requerido}
            value={value}
            onChange={(e) => setRespuestas({ ...respuestas, [campo.nombre]: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona una opción</option>
            {campo.opciones?.map((opcion, i) => (
              <option key={i} value={opcion}>{opcion}</option>
            ))}
          </select>
        )
      case 'checkbox':
        return (
          <div className="flex items-center">
            <input
              type="checkbox"
              checked={value}
              onChange={(e) => setRespuestas({ ...respuestas, [campo.nombre]: e.target.checked })}
              className="mr-2"
            />
            <label className="text-sm">Acepto</label>
          </div>
        )
      default:
        return (
          <input
            type={campo.tipo}
            required={campo.requerido}
            value={value}
            onChange={(e) => setRespuestas({ ...respuestas, [campo.nombre]: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder={campo.placeholder}
          />
        )
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando formulario...</p>
      </div>
    )
  }

  if (!formulario) {
    return (
      <div className="text-center p-8">
        <h1 className="text-2xl font-bold mb-4">Formulario no encontrado</h1>
        <p className="text-gray-600">El formulario que buscas no existe o está inactivo.</p>
      </div>
    )
  }

  return (
    <div className="max-w-2xl mx-auto">
      <div className="bg-white p-6 rounded-lg shadow-lg mt-8">
        <h1 className="text-3xl font-bold mb-2">{formulario.titulo}</h1>
        {formulario.descripcion && (
          <p className="text-gray-600 mb-6">{formulario.descripcion}</p>
        )}

        <form onSubmit={handleSubmit} className="space-y-6">
          {formulario.campos?.map((campo) => (
            <div key={campo.id}>
              <label className="block text-sm font-medium mb-2">
                {campo.etiqueta}
                {campo.requerido && <span className="text-red-500 ml-1">*</span>}
              </label>
              {renderCampo(campo)}
            </div>
          ))}

          <div>
            <label className="block text-sm font-medium mb-2">
              Fecha de visita (opcional)
            </label>
            <input
              type="date"
              value={fechaVisita}
              onChange={(e) => setFechaVisita(e.target.value)}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
          </div>

          <button
            type="submit"
            disabled={submitting}
            className="w-full bg-blue-600 text-white px-6 py-3 rounded hover:bg-blue-700 disabled:opacity-50 transition font-medium"
          >
            {submitting ? 'Enviando...' : 'Enviar Formulario'}
          </button>
        </form>
      </div>
    </div>
  )
}
